﻿Function New-MyScheduledTask {
    <#
            .SYNOPSIS
            This script can be used to create a Scheduled Task (Job) using a select group of the available parameters.

            .DESCRIPTION
            Creates a scheduled task to run a PowerShell script.
            See Get-Help Register-ScheduledJob -Full for more details about the underlying cmdlet.
            The scheduled task created, will run a PowerShell script in a 64Bit process. 

            .EXAMPLE
            The following will create a new Scheduled Task named 'RunMyPowerShellScript' which will run a PowerShell script every day at 5:00pm. It will also run immediately upon creation.
            
            $Splat_Task = @{
                ScriptFilePath = 'D:\powershell\test_task.ps1'
                TaskName = 'RunMyPowerShellScript'
                Trigger = ( New-JobTrigger -Daily -At "5:00:00 PM" )
                RunNow = $true
            }
            
            New-MyScheduledTask @Splat_Task -Verbose

    #>
    [CmdletBinding()]
    Param(
    
        <#
                Specifies values for the parameters of the script that is specified by the FilePath parameter.
        #>  
        [Parameter(
                Mandatory = $false )]  
        $ArgumentList,
        
        <# 
                Specifies the mechanism that is used to authenticate the user's credentials. 
                The acceptable values for this parameter are: Basic, Credssp, Default, Digest, Kerberos, Negotiate, NegotiateWIthImplicitCredential
        #>
        [ValidateSet('Basic', 'Credssp', 'Default', 'Digest', 'Kerberos', 'Negotiate', 'NegotiateWIthImplicitCredential', IgnoreCase = $true)]
        [Parameter(
                Mandatory = $false )]
        $Authentication = 'Default', 
        
        <# 
                Specifies a user account that has permission to run the scheduled job. The default is the current user.
    
                Type a user name, such as User01 or Domain01\User01, or enter a PSCredential object, such as one from the Get-Credential cmdlet. If you 
                enter only a user name, you are prompted for a password.
        #>
        [Parameter(
                Mandatory = $false )]  
        [PSCredential] $Credential, 
        
        <# 
                Specifies a script that the scheduled job runs. Enter the path to a .ps1 file on the local computer. To specify default values for the 
                script parameters, use the ArgumentList parameter. Every Register-ScheduledJob command must use either the ScriptBlock or FilePath 
                parameters.
        #>
        [ValidateScript({ If( Test-Path -Path $_ -PathType Leaf ){ $true } else { Throw "Invalid path given: $_" } })]
        [Parameter(
                ParameterSetName = 'FilePath',
                Mandatory = $true)]
        [string]$ScriptFilePath, 
         
        <# 
                Specifies a name for the scheduled job. The name is also used for all started instances of the scheduled job. 
                The name must be unique on the computer. This parameter is required. 
        #>
        [Parameter(
                Mandatory = $true )]
        [string]$TaskName, 
        
        <#
                Runs the scheduled job in a 32-bit process.
        #>
        [Parameter(
                Mandatory = $false )]
        [switch] $RunAs32,
        
        <#
                Specifies how often to run the job.
                @{Text=}
        #>
        [Parameter(
                Mandatory = $false )]
        [TimeSpan] $RunEvery,
       
        <#
                Starts a job immediately, as soon as the Register-ScheduledJob cmdlet is run. This parameter eliminates the need to trigger Task 
                Scheduler to run a Windows PowerShell script immediately after registration, and does not require users to create a trigger that 
                specifies a starting date and time.
        #>
        [Parameter(
                Mandatory = $false )]
        [switch] $RunNow,
    
        <#
                Sets options for the scheduled job. Enter a ScheduledJobOptions object, such as one that you create by using the New-ScheduledJobOption 
                cmdlet, or a hash table value.
                
                You can set options for a scheduled job when you register the schedule job or use the Set-ScheduledJobOption or Set-ScheduledJob cmdlets 
                to change the options.
                
                Many of the options and their default values determine whether and when a scheduled job runs. Be sure to review these options before 
                scheduling a job. For a description of the scheduled job options, including the default values, see New-ScheduledJobOption.
                
                To submit a hash table, use the following keys. In the following hash table, the keys are shown with their default values.
                
                `@{StartIfOnBattery=$False; StopIfGoingOnBattery=$True; WakeToRun=$False; StartIfNotIdle=$False; IdleDuration="00:10:00"; 
                IdleTimeout="01:00:00"; StopIfGoingOffIdle=$True; RestartOnIdleResume=$False; ShowInTaskScheduler=$True; RunElevated=$False; 
                RunWithoutNetwork=$False; DoNotAllowDemandStart=$False; MultipleInstancePolicy=IgnoreNew}`
        #>
        [Parameter(
                Mandatory = $false )]
        [object[]] $ScheduledJobOption,   
    
        <# 
                Specifies the triggers for the scheduled job. Enter one or more ScheduledJobTrigger objects, such as the objects that the New-JobTrigger 
                cmdlet returns, or a hash table of job trigger keys and values.
                
                A job trigger starts the schedule job. The trigger can specify a one-time or recurring scheduled or an event, such as when a user logs on 
                or Windows starts.
                
                The Trigger parameter is optional. You can add a trigger when you create the scheduled job, use the Add-JobTrigger, Set-JobTrigger, or 
                Set-ScheduledJob cmdlets to add or change job triggers later, or use the Start-Job cmdlet to start the scheduled job immediately. You can 
                also create and maintain a scheduled job without a trigger that is used as a template.
                
                To submit a hash table, use the following keys:
                
                `@{Frequency="Once"` (or Daily, Weekly, AtStartup, AtLogon); `At="3am"` (or any valid time string);  `DaysOfWeek="Monday", "Wednesday"` 
                (or any combination of day names);  `Interval=2` (or any valid frequency interval);  `RandomDelay="30minutes"` (or any valid timespan 
                string);  `User="Domain1\User01"` (or any valid user; used only with the AtLogon frequency value) }
        #>
        [Parameter(
                Mandatory = $false )]
        [object[]] $Trigger

    )
    
    BEGIN {
        # Deliberately, left empty.
    }
    
    PROCESS {
    
    
        # Register-ScheduledJob creates a new scheduled task within the 
        #   Task Scheduler Library \ Microsoft \ Windows \ PowerShell \ ScheduledJobs
        #
        #   Job definition information is located in the 
        #     C:\Users\<USERNAME>\AppData\Local\Microsoft\Windows\PowerShell\ScheduledJobs
        #     Example: C:\Users\brian.flaherty\AppData\Local\Microsoft\Windows\PowerShell\ScheduledJobs
    
        #  You can use the following to open the Task Scheduler application
        # taskschd.msc
    
        <# 
                Parameter List
                --------------------
                $ArgumentList
                $Authentication
                $Credential
                * $ScriptFilePath
                * $TaskName
                $RunEvery
                $RunNow
                $ScheduledJobOption
                $Trigger
        #>
    
    
        Try {
            $NewScheduledJob = @{
                Name = $TaskName
                FilePath = $ScriptFilePath
            }
    
            if($ArgumentList) {
                $NewScheduledJob.ArgumentList = $ArgumentList
            }
            
            if($Authentication) {
                $NewScheduledJob.Authentication = $Authentication
            }  
              
            if($Credential) {
                $NewScheduledJob.Credential = $Credential
            } 
            
            if($RunEvery) {
                $NewScheduledJob.RunAs32 = $RunAs32
            }  
              
            if($RunEvery) {
                $NewScheduledJob.RunEvery = $RunEvery
            }             
              
            if($RunNow) {
                $NewScheduledJob.RunNow = $RunNow
            }
                      
            if($ScheduledJobOption) {
                $NewScheduledJob.ScheduledJobOption = $ScheduledJobOption
            }           
                           
            if($Trigger) {
                $NewScheduledJob.Trigger = $Trigger
            }
    
            Register-ScheduledJob @NewScheduledJob
        
        } Catch {
            Write-Error -Message "Failed to create new scheduled job."
        }
    }
    
    END {
        # Deliberately, left empty.
    }
}